
/**
 *  @file
 *  Attach Media WYSIWYG behaviors.
 */

(function ($) {

Drupal.media = Drupal.media || {};

/**
 * Register the plugin with WYSIWYG.
 */
Drupal.wysiwyg.plugins.media = {

  /**
   * The selected text string.
   */
  selectedText: null,

  /**
   * Determine whether a DOM element belongs to this plugin.
   *
   * @param node
   *   A DOM element
   */
  isNode: function(node) {
    return $(node).hasClass('media-element');
  },

  /**
   * Execute the button.
   *
   * @param data
   *   An object containing data about the current selection:
   *   - format: 'html' when the passed data is HTML content, 'text' when the
   *     passed data is plain-text content.
   *   - node: When 'format' is 'html', the focused DOM element in the editor.
   *   - content: The textual representation of the focused/selected editor
   *     content.
   * @param settings
   *   The plugin settings, as provided in the plugin's PHP include file.
   * @param instanceId
   *   The ID of the current editor instance.
   */
  invoke: function (data, settings, instanceId) {
    var insert, mediaInstance, $placeholder;
    if (data.format == 'html') {
      insert = new InsertMedia(instanceId);
      // CKEDITOR module support doesn't set this setting
      if (typeof settings['global'] === 'undefined') {
        settings['global'] = {id: 'media_wysiwyg'};
      }
      if (this.isNode(data.node)) {
        // Edit existing media instance.
        $placeholder = $(data.node);
        mediaInstance = Drupal.media.filter.getMediaInstanceFromElement($placeholder);
        // Feed media instance about current state in placeholder.
        mediaInstance.setPlaceholderFromWysiwyg($placeholder);
        Drupal.media.filter.addMediaInstance(mediaInstance);
        insert.onSelect([mediaInstance.getSettings()]);
      }
      else {
        // Store currently selected text.
        this.selectedText = data.content;

        // Insert new media.
        insert.prompt(settings.global);
      }
    }
  },

  /**
   * Attach function, called when a rich text editor loads.
   * This finds all [[tags]] and replaces them with the html
   * that needs to show in the editor.
   *
   * This finds all JSON macros and replaces them with the HTML placeholder
   * that will show in the editor.
   */
  attach: function (content, settings, instanceId) {
    content = Drupal.media.filter.replaceTokenWithPlaceholder(content);
    return content;
  },

  /**
   * Detach function, called when a rich text editor detaches
   */
  detach: function (content, settings, instanceId) {
    content = Drupal.media.filter.replacePlaceholderWithToken(content);
    return content;
  }
};
/**
 * Defining InsertMedia object to manage the sequence of actions involved in
 * inserting a media element into the WYSIWYG.
 * Keeps track of the WYSIWYG instance id.
 */
var InsertMedia = function (instance_id) {
  this.instanceId = instance_id;
  return this;
};

InsertMedia.prototype = {
  /**
   * Prompt user to select a media item with the media browser.
   *
   * @param settings
   *    Settings object to pass on to the media browser.
   *    TODO: Determine if this is actually necessary.
   */
  prompt: function (settings) {
    Drupal.media.popups.mediaBrowser($.proxy(this, 'onSelect'), settings);
  },

  /**
   * On selection of a media item, display item's display configuration form.
   */
  onSelect: function (mediaSettings) {
    this.mediaSettings = mediaSettings[0];
    Drupal.media.popups.mediaStyleSelector(this.mediaSettings, $.proxy(this, 'insert'), {});
  },

  /**
   * When display config has been set, insert the placeholder markup into the
   * wysiwyg and generate its corresponding json macro pair to be added to the
   * tagmap.
   */
  insert: function (formatted_media) {
    var mediaInstance = new Drupal.media.WysiwygInstance($.extend({}, formatted_media.options, {
      attributes: this.mediaSettings.attributes,
      link_text: Drupal.wysiwyg.plugins.media.selectedText
    }), formatted_media.html);
    // Register the instance so it's not lost.
    Drupal.media.filter.addMediaInstance(mediaInstance);
    // Insert placeholder markup into wysiwyg.
    Drupal.wysiwyg.instances[this.instanceId].insert(mediaInstance.getPlaceholderHtml());
  }
};

})(jQuery);
